unit Common;
{

===============================================
             VANTEK COMMON ROUTINES
===============================================
This unit contains ton's of useful interface,
formatting, and other routines.

Used by almost everything Vantek does.
All routines can be used by Public Domain,
and are released into the Public Domain as free
source.

Vantek cannot be held resposible for this
could when used in other products.

Copyright (c) 1997-1999 Vantek Corp.
--------------------------------------------------------------------
GPL INFO:
--------------------------------------------------------------------
This file is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This file is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this file; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

interface
uses
	Graphics, TypInfo,
	Dialogs, WinTypes, WinProcs, iniFiles, Printers,
    Messages, Forms, classes, SysUtils;

    {misc routines}
    function DirExists(dir: string): boolean;
    procedure RemoveMDX(dbfile: String);
    function ColorPrinter: boolean;
    function GetPrinterIndex(prname: string): integer;
    function DotProd(x1, x2, y1, y2: extended): extended;
    procedure WriteFont(fnt: TFont; ini: TiniFile; sect, key: string);
    procedure ReadFont(fnt: TFont; ini: TiniFile; sect, key: string);
    function WinNT: boolean;
    function IsValidIP(inp: string): boolean;

    {TList Routines}
    procedure ClearList(lst: TList);

    {form position handlers}
	procedure SetPos(Form: TForm; iniFile: String);
	procedure SavePos(Form: TForm; inifile: string);

    {misc window routines}
	procedure SaveHistory(filename, section: String; tl: TStringList);
    procedure CenterChild(frm, Main: TForm; bbheight: integer);
	{function GetHistory(filename, section: string): TStringList;}
    procedure GetHistory(filename, section: string; tsl: TStringList);
	function RunCmd(cmdin: string): integer;
    function UNCPath(inp: string): string;
    procedure DelINIKey(fname, sect, key: string);
    procedure DoPrevInstance(winclass, appname, inifile: string; msg: integer);

    {boolean formatting routines...}
    function BoolToStr(inp: boolean): string;
	function StrToBool(inp: string): boolean;
	function BoolToInt(inp: boolean): integer;
	function IntToBool(inp: integer): boolean;
    function BoolToPass(inp: boolean): string;

    {string handling routines}
	function FillStr(num: integer; ic: char): string;
	function RTrim(inp: string): string;
	function LTrim(inp: string): string;
    function TrimPath(inp: string): string;
    function GetStringHeight(HDC: Integer; inp: string): integer;

    {$ifdef Windows}
    function Trim(inp: string): string;
    function CopyFile(src, dest: string): boolean;
    function ExtractFileDrive(path: string): string;
    function DriveNum(path: string): integer;
    {$endif}

    function LeftChar(instring: string; nchar: word): string;
    function RightChar(instring: string; nchar: word): string;
    function qtrim(instring: string): string;
    function Dots2Bars(inp: string): string;
    function FilterNums(inp: string): string;
    procedure pparse(ch, inp: string; lst: TStringList);
    procedure pparsebuff(ch: string; inp: pchar; buffsize: integer; lst: TStringList);
    function AggStringList(sl, sl2: TStringlist): string;
    Procedure CryptStr(var s: string);

    {
    math functions,
    rounding functions,
    numerical display functions
    }

    function formatd(inp: extended): string;
    function formatd2(inp: extended): string;
	function Radians(deg: real): real;
    function SToReal(inp: string): real;
    function SToInt(inp: string): integer;
    function RoundReal(inp: real; places: word): Real;
    function RoundRes(inp, res: real): Real;
    function RealToStr(inp: real; places: word): string;
    function pow(base, e: real): real;
    function tan(x: Real): real;
    function RadToDeg(x: Real): real;
    function DegToRad(x: real): real;
    procedure PolarToCart(var x,y: extended);
    function DMSDisplay(x: real): string;
    function DMSValue(inp: string): extended;
    function LongDMSDisplay(x: real): string;
    function dround(inval: real; digits: integer): real;
    function TimeDisplay(x: real): string;
    function UnixTimeToDT(x: longint): TDateTime;
    function GetFraction(inp: string): real;
    procedure CartToSphere(x,y,z: extended; var r,a1,a2: extended);
    Function NlogX(X: Real; N:Real): Real;
    function IntToBinStr(inp: integer): string;

    {Other Trig Functions}
    function ArcSin(x: extended): extended;
    function ArcCos(x: extended): extended;

    {printer functions}
    function GetDefaultPrinter: string;
    function SetDefaultPrinter(prn: string): boolean;
    function GetPrinterInfo(prn: string): string;
    procedure GetPrinters(var prnlst: TStringList);
    procedure GetDevModeInfo(info: TStringList);
    function ResetPrinter(scale, res: integer): boolean;
    procedure SetPrinterPage(PaperWidth, PaperHeight: integer);
    procedure SetPrinterIndex(Index: Integer);

    {Date Functions}
    function NameofMonth(mo: integer): string;
    function NumDaysInMonth(mo, yr: integer): integer;
    function ShortMonthIndex(inp: string): word;
    function BigStrToDateTime(inp: string): TDateTime;
    function DayIndex(inp: string): integer;
    function y2kadjust(inp: TDatetime): TDateTime;
    procedure y2kFormat;

    function GetVersionInfo(KeyList, ValList: TStringList): string;

const
    Date1970 = 25569;       {date constants for Y2K stuff}
    DateY2K = 36524;
    Epsilon = 0.0000001;    {Eps constant for float compares}

var
	MainINI: string;

{---------------------------------------}
{---------------------------------------}
{---------------------------------------}
implementation

{---------------------------------------}
procedure SetPos(Form: TForm; inifile: String);
var
	ini: TIniFile;
    vals: string;
    valc: string;
    key: string;
    p1, p2, p3: integer;
    t, l, w, h: integer;
    Code: integer;
begin
	{
    This functionsets the position of a form
    based on coords from ini file. All of these
    entries goto the section named [POS]
    and use a format of:
    C:\FOO\FOOBAR.EXE:Window Caption=top/left/width/height
    }
    if trim(inifile) = '' then exit;

	ini := TIniFile.Create(inifile);
    key := Application.EXEName + ':' + UpperCase(Form.Caption);
    vals := ini.ReadString('POS', key, '');
    valc := vals;
    ini.Free;

	p1 := pos('/', vals);
    if p1 = 0 then exit;
    if p1 > 0 then delete(vals, 1, p1);
    p2 := pos('/', vals) + p1;
    if p2 > 0 then delete(vals, 1, p2);
    p3 := pos('/', vals) + p1 + p2;

    if p1 < 0 then p1 := 0;
    if p2 < 0 then p2 := 0;
    if p3 < 0 then p3 := 0;

    Val(Copy(valc, 1, p1-1), t, Code);
    Val(Copy(valc, p1+1, p2-p1-1),l, Code);
	Val(Copy(valc, p2+1, P3-p2-1),w,Code);
    Val(Copy(valc, p3+1, length(valc)-p3),h,Code);
    Form.SetBounds(l,t,w,h);
end;

{---------------------------------------}
procedure SavePos(Form: TForm; inifile: string);
var
	ini: TINIFile;
    key: string;
    vals: string;
begin
    {
    Save the coordinates of a form to an INI
    file. Use the [POS] section. See the SetPos
    function for format details.
    }
	if form.windowstate = wsMinimized then exit;
    if Trim(iniFile) = '' then exit;

    ini := TINIFile.Create(inifile);
    key := Application.EXEName + ':' + Uppercase(Form.Caption);
    vals :=
    	Format('%d', [Form.Top]) + '/' +
        Format('%d', [Form.Left]) + '/' +
        Format('%d', [Form.Width]) + '/' +
        Format('%d', [Form.Height]);
    ini.WriteString('POS', key, vals);
    ini.Free;
end;

{---------------------------------------}
function BoolToStr(inp: boolean): string;
begin
	{ returns 'True' if true, 'False' if false}
    if inp then BoolToStr := 'True' else BoolToStr := 'False';
end;

{---------------------------------------}
function BoolToPass(inp: boolean): string;
begin
	{returns either 'PASS' or 'FAIL' from a bool}
	if inp = true then Result := 'PASS' else Result := 'FAIL';
end;

{---------------------------------------}
function StrToBool(inp: string): boolean;
begin
	{cheks for 'TRUE' and returns a bool}
	if Trim(Uppercase(inp)) = 'TRUE' then StrToBool := true else StrToBool := false;
end;

{---------------------------------------}
function BoolToInt(inp: boolean): integer;
begin
	{returns -1 for True values, 0 for false values}
	if inp then BoolToInt := -1 else BoolToInt := 0;
end;

{---------------------------------------}
function IntToBool(inp: integer): boolean;
begin
	{returns false for 0 values, true for all others}
	if inp = 0 then IntToBool := false else IntToBool := true;
end;

{---------------------------------------}
procedure pparse(ch, inp: string; lst: TStringList);
var
    tmps, tmp2: string;
    cp: integer;
begin
	{
    parse the incoming string (inp) based
    on the character (ch). Return each
    token, or item in the TStringList
    passed to use. Make sure the TStringList
    already exists.
    }
	lst.clear;
    cp := pos(ch, inp);
    tmp2 := inp;
    while cp > 0 do begin
        tmps := Copy(tmp2, 1, (cp-1));
        {tmp2 := Copy(tmp2, cp + 1, Length(tmp2)-cp);}
        tmp2 := LTrim(Copy(tmp2, cp + 1, Length(tmp2)-cp));
        lst.add(tmps);
        cp := pos(ch, tmp2);
        end;
    lst.add(tmp2);
end;

{---------------------------------------}
procedure pparsebuff(ch: string; inp: pchar; buffsize: integer; lst: TStringList);
var
    pStart, pEnd, pBuff: PChar;
    pToken: PChar;
    split: pchar;
begin
	{
    parse the incoming buffer (inp) based
    on the character (ch). Return each
    token, or item in the TStringList
    passed to use

    pgm 3/11/99 - completely rewritten
    to solve the invalid pointer operations & such.
    also fixes mem. leaks
    }
    pBuff := StrAlloc(buffsize + 1);
    split := StrAlloc(1);
    pToken := StrAlloc(buffsize + 1);

    StrCopy(pBuff, inp);
    StrPCopy(split, ch);
    pStart := pBuff;
	lst.clear;
    pEnd := StrPos(pStart, split);
    while pEnd <> nil do begin
        lst.add(Copy(StrPas(pBuff), pStart - pBuff + 1, pEnd - pStart));
        pStart := pEnd + 1;
        pEnd := StrPos(pStart, split);
        end;
    lst.add(StrPas(pStart));
    StrDispose(split);
    StrDispose(pToken);
    StrDispose(pBuff);
end;

{---------------------------------------}
function AggStringList(sl, sl2: TStringlist): string;
var
    tmps: string;
    i: integer;
begin
    {
    this function returns a large string..
    each line contains a single item from the list,
    followed by a CR+LF character.

    Pass a second TStringList to get a 'values'
    type of list like this:
        sl[0]=sl2[0]
    }
    tmps := '';
    for i := 0 to sl.Count - 1 do begin
        tmps := tmps + sl[i];
        if sl2 <> nil then
            tmps := tmps +  ': ' + sl2[i];
        tmps := tmps + chr(13) + Chr(10);
        end;
    Result := tmps;
end;


{---------------------------------------}
function LTrim(inp: string): string;
var
	tmps: string;
    i: integer;
    ex: boolean;
begin
	{trim all left hand whitespace off of a string}
	tmps := '';
    ex := false;
    for i := 1 to length(inp) do begin
        if ((inp[i] <> ' ') and (inp[i] <> chr(9))) or (ex = true) then begin
        	tmps := tmps + inp[i];
            ex := true;
            end;
    	end;
    Result := tmps;
end;

{---------------------------------------}
function RTrim(inp: string): string;
var
   tmps: string;
   i: integer;
begin
	{trim off right hand trailing spaces}
   	tmps := inp;
   	for i := length(inp) downto 1 do begin
        if inp[i] = ' ' then begin
            tmps := copy(inp, 1, i-1);
            end
        else begin
        	Result := tmps;
            exit;
            end;
    	end;
    Result := tmps;
end;

{---------------------------------------}
function formatd(inp: extended): string;
var
	v: extended;
	tmps: string;
begin
	{
    returns a the double prec number formatted string
    Returns 6 decimal precision for 0 -> 999
    Returns 3 decimal precision for 1000 -> 100000
    Returns 2 decimal precision for 100000 +
    }
    try
    	v := inp;
    	if abs(v) < 0.00000000001 then v := 0.0000000;
        tmps := LTrim(format('%12.6f', [v]));
        if inp >= 100000 then
            tmps := LTrim(format('%12.2f', [v]))
        else if inp >= 1000 then
            tmps := LTrim(format('%12.3f', [v]));
        if inp >= 0.0 then
            tmps := '+' + tmps;
        if tmps = '0.000000' then tmps := '+0.000000';
    except
    	tmps := '0.0';
    end;
	Result := tmps;
end;

{---------------------------------------}
function formatd2(inp: extended): string;
var
	tmps: string;
begin
	{ same as formatd but doesn't print a + sign for positives}
    try
    	if abs(inp) < 0.00000000001 then inp := 0.0000000;
        tmps := LTrim(format('%12.6f', [inp]));
        if inp >= 100000 then
            tmps := LTrim(format('%12.2f', [inp]))
        else if inp > 1000 then
            tmps := LTrim(format('%12.3f', [inp]));
    except
    	tmps := '0.0';
    end;
	Result := tmps;
end;

{---------------------------------------}
function qtrim(instring: string): string;
var
	tmps: string;
begin
	{strip off first and last " characters}
    tmps := instring;
    if leftchar(tmps, 1) = '"' then
    	tmps := RightChar(tmps, length(tmps) - 1);
    if RightChar(tmps, 1) = '"' then
    	tmps := LeftChar(tmps, Length(tmps) - 1);
    Result := tmps;
end;

{---------------------------------------}
function Dots2Bars(inp: string): string;
var
	ch, tmps: string;
    i: integer;
begin
	{strip all '.' and conver to '_'}
    tmps := inp;
    Result := '';
    for i := 1 to Length(inp) do begin
    	ch := Copy(inp, i, 1);
    	if  ch = '.' then
            Result := Result + '_'
        else
        	Result := Result + ch;
    	end;
end;

{---------------------------------------}
{pgm 3/11/99 - converted to a proc to eliminate mem leak}
procedure GetHistory(filename, section: string; tsl: TStringList);
var
	i: integer;
    tl: TStringList;
    ini: TiniFile;
begin
	{
    get the history list from an INI file.
    this routine may be slightly funky because of the
    TStringList created and passed back
    Memory leaks may occur.

    pgm 3/11/99 - turned this into a procedure,
    the stringlist should already exist and be
    passed into this function... eliminates the mem leaks.
    }
	tl := TStringList.Create;
	ini := TiniFile.Create(filename);
    ini.ReadSectionValues(section, tl);
    tsl.Clear;
    for i := 0 to tl.count - 1 do begin
        if i < 10 then
            tsl.Add(tl.Values[IntToStr(i+1)]);
        end;
    tl.free;
    ini.free;
end;

{---------------------------------------}
function UNCPath(inp: string): string;
var
	tmps: string;
begin
	{$ifdef Win32}
	tmps := ExtractFilePath(ExpandUNCFileName(inp + '\foobar.txt'));
    {$else}
    {16 bit windows does not support UNC path names well..}
    tmps := inp;
    {$endif}

    tmps := Trim(tmps);
    if Copy(tmps, Length(tmps), 1) = '\' then
    	tmps := LeftChar(tmps, Length(tmps)-1);

    Result := tmps;
end;

{---------------------------------------}
procedure SaveHistory(filename, section: String; tl: TStringList);
var
	i: integer;
    ini: TiniFile;
begin
	ini := TiniFile.Create(filename);
    ini.EraseSection(section);
    for i := 0 to tl.Count - 1 do
    	ini.WriteString(section, IntToStr(i + 1), tl.Strings[i]);
    {pgm 3/11/99 - mem leak}
    ini.Free;
end;

{---------------------------------------}
function RunCmd(cmdin: string): integer;
var
	cmd: Array [0..200] of Char;
begin
	{run a specified command line using WinExec}
    {We need to add 32 bit functionality using CreateProcess here}
	StrPCopy(cmd, cmdin);
	RunCmd := WinExec(cmd, SW_SHOWNORMAL);
end;

{---------------------------------------}
function FillStr(num: integer; ic: char): string;
var
	i: integer;
    tmps: string;
begin
	{return a string a num characters...}
    tmps := '';
    for i := 1 to num do
    	tmps := tmps + ic;

    FillStr := tmps;
end;

{---------------------------------------}
function Radians(deg: real): real;
begin
    {convert degrees into radians}
    Result := DegToRad(deg);
end;

{---------------------------------------}
function SToReal(inp: string): real;
var
	tmpe: extended;
begin
	{
    protected function to convert strings to floats
    will never GPF since the internal try/except..
    If we get an error, the function returns 0.0
    This CAN lead to some strange behavior, but
    provides a somewhat more bullet-proof
    StrToFloat function.
    }
	try
    	tmpe := StrToFloat(inp);
    except on EConvertError do
    	tmpe := 0.0;
    end;
    Result := tmpe;
end;

{---------------------------------------}
function SToInt(inp: string): integer;
var
	tmpi: integer;
begin
	{
    protected function to convert ints..
    See SToReal for more info.
    }
	try
    	tmpi := StrToInt(inp);
    except on EConvertError do
    	tmpi := 0;
    end;
    Result := tmpi;
end;


{---------------------------------------}
procedure CenterChild(frm, Main: TForm; bbheight: integer);
begin
	{
    center the childform inside a parent, taking
    into account a toolbar height (bbheight)
    }
	frm.Left := (Main.ClientWidth - frm.Width) div 2;
    frm.Top := (Main.ClientHeight - bbheight - frm.Height) div 2;
end;

{---------------------------------------}
function pow(base, e: real): real;
begin
	{return base rfaised to the power of e}
	result := exp(e * ln(base));
end;

{---------------------------------------}
function tan(x: Real): real;
begin
	{calculate tangent function - x is in Radians}
	result := sin(x) / cos(x);
end;

{---------------------------------------}
function dround(inval: real; digits: integer): real;
var
    wnum: longint;
    factor, fracnum: extended;
begin
    {
    round a floating point number to a specific number
    of digits. This the "ANSI" way of doing things
    }
    factor := pow(10.0, digits);
    wnum := trunc(inval * factor);
    fracnum := inval - (wnum / factor);
    if (fracnum > 0.499999) and (fracnum < 0.500001) then begin
    	if (wnum and 1) = 0 then
        	Result := wnum / factor
        else
        	Result := (wnum + 1) / factor;
        end
    else
        Result := (trunc((inval * factor) + 0.5)) / factor;
end;

{---------------------------------------}
function RightChar(instring: string; nchar: word): string;
var
	tmps: string;
begin
	{returns the rightmost n characters of a string}
    tmps := Copy(instring, length(instring) - nchar + 1, nchar);
    Result := tmps;
end;

{---------------------------------------}
function LeftChar(instring: string; nchar: word): string;
var
	tmps: string;
begin
	{returns the leftmost n characters of a string}
    tmps := Copy(instring, 1, nchar);
    Result := tmps;
end;

{---------------------------------------}
function RadToDeg(x: Real): real;
begin
	{returns an angle in degrees}
    Result := (360 / (2.0*Pi)) * x;
end;

{---------------------------------------}
function DegToRad(x: real): real;
begin
	{returns an angle in radians}
    Result := x / (360 / (2.0 * Pi));
end;

{---------------------------------------}
function IntToBinStr(inp: integer): string;
var
    cv, res, rmd: integer;
    tmps: string;
begin
    {convert int to binary string}
    cv := inp;
    res := inp;
    while res > 0 do begin
        res := cv div 2;
        rmd := cv mod 2;
        if rmd > 0 then tmps := '1' + tmps  else tmps := '0' + tmps;
        cv := res;
    end;

    {pad with left zeros}
    while length(tmps) < 8 do
        tmps := '0' + tmps;
    Result := tmps;

end;

{---------------------------------------}
function RealToStr(inp: real; places: word): string;
var
	fs, tmps: string;
    w: word;
begin
	{returns a string converted from a real}
    w := places + 3;
    fs := '%' + IntToStr(w) + '.' + IntToStr(places) + 'f';
    tmps := Format(fs, [inp]);
    Result := tmps;
end;

{---------------------------------------}
function RoundReal(inp: real; places: word): Real;
var
	mult: longint;
    newval: real;
begin
	{round a real value to n places}
    {ie, 1.00007 --> 1.0001}
    mult := trunc(pow(10.0, places));
    newval := (round(inp * mult)) / mult;
    Result := newval;
end;

{---------------------------------------}
function RoundRes(inp, res: real): Real;
var
    newval: real;
begin
	{
    round a value to a display resolution,
    ie 0.0002 or 0.0005
    }
    newval := trunc((inp / res) + 0.5) * res;
    Result := newval;
end;

{---------------------------------------}
function DMSDisplay(x: real): string;
var
	xp, cval: real;
	rot, deg, min, sec: Integer;
	tmps: string;
    neg: boolean;
begin
	{
    show the input (in decimal degrees)
    in degrees, minutes, seconds
    using a decimal seperator:  108.34.10
    }
    if x = 0.0 then begin
    	{Result := '  N/A';}
        Result := '0.00.00';
        exit;
        end;

    neg := false;
    if x < 0 then begin
    	neg := true;
        xp := abs(x);
        end
    else
    	xp := x;

    deg := trunc(xp);
    if deg >= 360 then begin
    	rot := Trunc(deg / 360);
        deg := deg - (rot * 360);
        end;
    cval := xp - deg;
    min := trunc(cval * 60);
    cval := cval - (min / 60);
    sec := trunc(cval * 3600);
    if neg then deg := - deg;
    tmps := Format('%3d', [deg]) + '.' + Format('%2.2d', [min]) + '.' + Format('%2.2d', [sec]);
    Result := tmps;
end;

{---------------------------------------}
procedure PolarToCart(var x,y: extended);
var
	r,a: extended;
begin
	{
    Convert polar coords to cart.
    Assume Dec Degrees, not Radians
    }
	r := x;
    a := y;

    x := r * cos(Radians(a));
    y := r * sin(Radians(a));
end;

{---------------------------------------}
procedure CartToSphere(x,y,z: extended; var r,a1,a2: extended);
begin
	{
    Converts x,y,z coords to spherical
    NOTE -> Returns angles in Radians, not Degrees.
    }

	r := sqrt(x*x + y*y + z*z);
    a1 := ArcTan(y / x);
    a2 := ArcCos( z / r);
end;

{---------------------------------------}
function ArcSin(x: extended): extended;
begin
	Result := ArcTan (x/sqrt (1-sqr (x)));
end;

{---------------------------------------}
function ArcCos(x: extended): extended;
begin
	Result := ArcTan (sqrt (1-sqr (x)) / x);
end;

{---------------------------------------}
function DMSValue(inp: string): extended;
var
	p1, p2: integer;
    deg, min, sec: string;
begin
	{return the dec. degree equiv..}
    p1 := pos('.', inp);
    p2 := pos('.', RightChar(inp, length(inp) - p1)) + p1;

    Result := 0.0;

    if p1 = 0 then exit;
    if p2 = 0 then exit;

    deg := Trim(Copy(inp, 1, p1-1));
    min := Trim(Copy(inp, p1+1, p2-p1-1));
    sec := Trim(Copy(inp, p2+1, length(inp) - p2));
    if deg = '' then deg := '0';
    if min = '' then min := '0';
    if sec = '' then sec := '0';
    Result := (StrToInt(sec) / 3600) + (StrToInt(min) / 60) + StrToInt(deg);
end;

{---------------------------------------}
function LongDMSDisplay(x: real): string;
var
	cval: real;
    deg, min: word;
    tmps: string;
begin
	{
    show the input (in decimal degrees)
    in degrees, minutes, seconds
    using a string seperators:  108 deg  34 min  10 sec
    }
    if x = 0.0 then begin
    	Result := '  N/A';
        exit;
        end;

    deg := Trunc(x);
    cval := x - deg;
    min := Trunc(cval * 60);
    tmps := format('%3d', [deg]) + ' Deg. ' + format('%2d', [min]) + ' Min.';
    Result := tmps;
end;

{---------------------------------------}
function GetDefaultPrinter: string;
var
    ini: TiniFile;
    tmps: string;
    p1: integer;
begin
    {get the name of the current printer...}

    {
    NOTE: This is the procedure support by MS, even
    in NT 3.51 and higher systems. NT automatically
    maps the WIN.INI file entries to the proper
    locations in the system registry.
    }
    Printer.PrinterIndex := -1;
    ini := TiniFile.Create('WIN.INI');
    tmps := ini.ReadString('Windows', 'device', '');
    p1 := Pos(',', tmps);
    Result := LeftChar(tmps, p1 - 1);
    ini.Free;
end;

{---------------------------------------}
 procedure SetPrinterIndex(Index: Integer);
  var
    n,d,p: array[0..255] of Char;
    h: THandle;
  begin
    { Set printer index }
    Printer.PrinterIndex := Index;
    { Trick TPrinter into reloading the driver capabilities }
    Printer.GetPrinter(n,d,p,h);
    Printer.SetPrinter(n,d,p,0);
  end;

{---------------------------------------}
function SetDefaultPrinter(prn: string): boolean;
var
    fullinfo: string;
    ini: TiniFile;
begin
    {
    set the default, current printer
    get the full info from GetPrinterInfo
    }

    {
    NOTE: This is the procedure support by MS, even
    in NT 3.51 and higher systems. NT automatically
    maps the WIN.INI file entries to the proper
    locations in the system registry.
    }
    fullinfo := GetPrinterInfo(prn);
    if fullinfo = '' then begin
        Result := false;
        exit;
        end;
    ini := TiniFile.Create('WIN.INI');
    ini.WriteString('Windows', 'device', fullinfo);
    ini.Free;
    Printer.PrinterIndex := -1;
    SendMessage(HWND_BROADCAST, wm_WinIniChange, 0, 0);
    Result := true;
end;

{---------------------------------------}
function GetPrinterInfo(prn: string): string;
var
    ini: TiniFile;
    v1, v2: string;
begin
    {
    return the complete printer info string
        <printername>,<driver>,<port>
    given a printer name...
    }

    {
    NOTE: This is the procedure support by MS, even
    in NT 3.51 and higher systems. NT automatically
    maps the WIN.INI file entries to the proper
    locations in the system registry.
    }

    ini := TiniFile.Create('WIN.INI');
    v1 := ini.ReadString('devices', prn, '');
    ini.Free;
    if v1 <> '' then begin
    	v2 := prn + ',' + v1;
        v2 := Trim(v2);
        Result := v2;
        end
    else
        Result := '';
end;

{---------------------------------------}
procedure SetPrinterPage(PaperWidth, PaperHeight: integer);
var
    pDev, pDrv, pPort: pchar;
    DeviceMode:		THandle;
    PDeviceMode:	PDevMode;
begin
    {
    This procedure sets the page dimensions
    for the current page. Pass dimension in mm

    API units are in tenths of mm.
    }
    {$ifdef Win32}
    GetMem (pDev, 255);
    GetMem (pDrv, 255);
    GetMem (pPort, 255);
    Printer.GetPrinter(pDev, pDrv, pPort, DeviceMode);
    PDeviceMode := GlobalLock(DeviceMode);
    {PDeviceMode^.dmPaperSize := DMPAPER_USER + DMPAPER_TABLOID;}
    PDeviceMode^.dmPaperWidth := PaperWidth * 10;
    PDeviceMode^.dmPaperLength := PaperHeight * 10;
    {PDeviceMode^.dmFields := DM_PAPERSIZE + DM_PAPERWIDTH + DM_PAPERLENGTH;}
    PDeviceMode^.dmFields := DM_PAPERWIDTH + DM_PAPERLENGTH;
    GlobalUnlock(DeviceMode);
    Printer.SetPrinter(pDev, pDrv, pPort, DeviceMode);
    SendMessage(HWND_BROADCAST, WM_SETTINGCHANGE, 0, 0);
    FreeMem(pDev, 255);
    FreeMem(pDrv, 255);
    FreeMem(pPort, 255);
    {$endif}
end;

{---------------------------------------}
procedure GetPrinters(var prnlst: TStringList);
var
    ini: TiniFile;
begin
    {return a list of the available printers...}
    ini := TiniFile.Create('WIN.INI');
    ini.ReadSection('devices', prnlst);
    ini.Free;
end;

{---------------------------------------}
function DirExists(dir: string): boolean;
var
	attr: word;
	srchrec: TSearchRec;
begin
	{determine if the directory exists...}
    attr := $10 or $01 or $02;
    FindFirst(dir, attr, srchrec);
    if Trim(srchrec.name) <> '' then Result := true else Result := false;
end;

{---------------------------------------}
Procedure CryptStr(var s: string);
var
	x: byte;
begin
	{
    this encryption routine will encrypt plain text
    as well as decrypt a previously encrypted string
    Use the length of the string as the Random Number seed
    }
	RandSeed := Length(s);
    for x := 1 to length(s) do
    	s[x] := chr(ord(s[x]) xor (Random(128) or 128));
end;

{---------------------------------------}
procedure RemoveMDX(dbfile: String);
const
	Value: byte = 0;
var
	f: File of Byte;
    mdxfilename: string;
begin
	{
    This routine removes the index flag.
    This allows a DBF file to be opened
    even when the MDX files are corrupt, etc...
    It makes the DBF file believe that NO indexes
    are attached to the table
    }
	try
        AssignFile(F, dbFile);
        Reset(F);
        Seek(F, 28);
        Write(F, Value);
        Write(F, Value);
    finally
    	CloseFile(F);
    	mdxfilename := ChangeFileExt(dbFile, '.MDX');
    	If FileExists(mdxfilename) then DeleteFile(mdxfilename);
    end;
end;

{---------------------------------------}
function GetPrinterIndex(prname: string): integer;
var
    tmps, prs: string;
    i: integer;
begin
    {return the Printer Index for a printer containing the string}
    {return -1 if the printer doesn't exist}

    Result := -1;
    prs := Trim(Lowercase(prname));
    for i := 0 to Printer.Printers.Count - 1 do begin
        tmps := lowercase(Printer.Printers[i]);
        if pos(prs, tmps) > 0 then begin
            Result := i;
            break;
            end;
        end;
end;

{---------------------------------------}
function ColorPrinter: boolean;
var
	prnDevice:		PChar;
    prnDriver:		PChar;
    prnPort:		PChar;
    DeviceMode:		THandle;
    PDeviceMode:	PDevMode;
    colormode: 		integer;
begin
	{
    Checks to see if the default printer
    is color capable.
    }
    Result := false;
    try
    	{allocate mem for strings.}
    	GetMem (prnDevice, 255);
        GetMem (prnDriver, 255);
        GetMem (prnPort, 255);
        Printer.GetPrinter(prnDevice, prnDriver, prnPort, DeviceMode);
        if DeviceMode = 0 then exit;
        PDeviceMode := GlobalLock(DeviceMode);
        colormode := PDeviceMode^.dmColor;
		GlobalUnlock(DeviceMode);
        if colormode = DMCOLOR_COLOR then Result := true else Result := false;
    except
    end;

end;

{---------------------------------------}
function DotProd(x1, x2, y1, y2: extended): extended;
begin
	Result := ((x1 * x2) + (y1 * y2));
end;


{---------------------------------------}
function FilterNums(inp: string): string;
var
	wdata: string;
    ac, i: integer;
    dflag: boolean;
begin
	{
    function trims all ascii chrs but numbers,
    decimal points, comma's, and +/-

    Useful when getting numerical information
    from a data stream from some external device
    (eg, RS-232 inputs, DDE strings, etc..)
    }
    wdata := '';
    dflag := false;
    for i := 1 to length(inp) do
        begin
        ac := ord(inp[i]);
        if (ac < 43) or (ac > 57) or (ac = 44) then
                {bogus chars}
        else begin
        	if (dflag = true) and (ac < 48) then
                {do nothing.. another decimal point, + or - sign}
            else
            	wdata := wdata + inp[i];
            if ac = 46 then dflag := true;
            end;
        end;
    Result := wdata;
end;

{---------------------------------------}
procedure GetDevModeInfo(info: TStringList);
var
	prnDevice:		PChar;
    prnDriver:		PChar;
    prnPort:		PChar;
    DeviceMode:		THandle;
    PDeviceMode:	PDevMode;
begin
	{
    This function reads in all the props of a
    DevMode structure and stores them
    to a TStringList which is passed
    into the proc.

    Used for debugging Printer stuff.
    }

    {allocate mem for strings.}
    GetMem (prnDevice, 255);
    GetMem (prnDriver, 255);
    GetMem (prnPort, 255);

    {get the printer DM}
	Printer.GetPrinter(prnDevice, prnDriver, prnPort, DeviceMode);
    if DeviceMode = 0 then begin
        info.add('Could not load the Printer Driver!');
        end
    else begin
        PDeviceMode := GlobalLock(DeviceMode);
        with info do begin
        	add('dmSpecVersion -> ' + IntToStr(PDeviceMode^.dmSpecVersion));
            add('dmDriverVersion -> ' + IntToStr(PDeviceMode^.dmDriverVersion));
            add('dmSize -> ' + IntToStr(PDeviceMode^.dmSize));
            add('dmOrientation -> ' + IntToStr(PDeviceMode^.dmOrientation));
            add('dmPaperSize -> ' + IntToStr(PDeviceMode^.dmPaperSize));
            add('dmPaperLength -> ' + IntToStr(PDeviceMode^.dmPaperLength));
            add('dmPaperWidth -> ' + IntToStr(PDeviceMode^.dmPaperWidth));
            add('dmScale -> ' + IntToStr(PDeviceMode^.dmScale));
            add('dmCopies -> ' + IntToStr(PDeviceMode^.dmCopies));
            add('dmDefaultSource -> ' + IntToStr(PDeviceMode^.dmDefaultSource));
            add('dmPrintQuality -> ' + IntToStr(PDeviceMode^.dmPrintQuality));
            add('dmColor -> ' + IntToStr(PDeviceMode^.dmColor));
            add('dmDuplex -> ' + IntToStr(PDeviceMode^.dmDuplex));
            end;
        GlobalUnlock(DeviceMode);
        end;
    FreeMem(prnDevice, 255);
    FreeMem(prnDriver, 255);
    FreeMem(prnPort, 255);
end;

{---------------------------------------}
function ResetPrinter(scale, res: integer): boolean;
var
	prnDevice:		PChar;
    prnDriver:		PChar;
    prnPort:		PChar;
    DeviceMode:		THandle;
    PDeviceMode:	PDevMode;
begin
	{
    Reset the printer to "normal stuff"
    }
    GetMem (prnDevice, 255);
    GetMem (prnDriver, 255);
    GetMem (prnPort, 255);
    try
    	{allocate mem for strings.}
        Printer.PrinterIndex := -1;
        Printer.GetPrinter(prnDevice, prnDriver, prnPort, DeviceMode);
        if DeviceMode = 0 then begin
        	MessageDlg('Could not load the Printer Driver!', mtError, [mbOK], 0);
            Result := false;
            exit;
            end
        else begin
        	PDeviceMode := GlobalLock(DeviceMode);
            PDeviceMode^.dmScale := scale;
            PDeviceMode^.dmPrintQuality := res;
            PDeviceMode^.dmFields := DM_PRINTQUALITY + DM_SCALE;
            GlobalUnlock(DeviceMode);
            Printer.SetPrinter(prnDevice, prnDriver, prnPort, DeviceMode);
            Result := true;
            end;
    finally
    	FreeMem(prnDevice, 255);
        FreeMem(prnDriver, 255);
        FreeMem(prnPort, 255);
    end;
end;

{---------------------------------------}
function TimeDisplay(x: real): string;
var
	am: string;
	hrs, min, sec: integer;
    res: double;
begin
	{return a hh:mm:ss format string from a real 0.434746}
    {use AM/PM}
    hrs := trunc(x * 24);
    res := (x - (hrs / 24)) * 24 * 60;
    if hrs > 12 then begin
    	am := 'PM';
    	hrs := hrs - 12;
        end
    else
    	am := 'AM';
    min := trunc(res);
    res := res - min;
    sec := trunc(res * 60);
    Result := 	trim(IntToStr(hrs)) + ':' +
                format('%2.2d', [min]) + ':' +
                format('%2.2d', [sec]) + ' ' + am;
end;

{---------------------------------------}
function UnixTimeToDT(x: longint): TDateTime;
var
    days, hr, min, sec: integer;
    dt: TDatetime;
begin
    { Convert a unix style time (seconds) to a TDateTime variable }

    min := x div 60;
    hr := min div 60;
    days := hr div 24;

    hr := hr - (days * 24);
    min := min - (hr * 60);
    sec := x - (days * 86400) - (hr * 3600) - (min * 60);
    try
        dt := EncodeTime(hr, min, sec, 0);
        dt := dt + days;
        Result := dt;
    except
        Result := 0.0;
    end;

end;


{---------------------------------------}
function NameofMonth(mo: integer): string;
begin
	case mo of
    1: Result := 'January';
    2: Result := 'February';
    3: Result := 'March';
    4: Result := 'April';
    5: Result := 'May';
    6: Result := 'June';
    7: Result := 'July';
    8: Result := 'August';
    9: Result := 'September';
    10: Result := 'October';
    11: Result := 'November';
    12: Result := 'December';
    else Result := 'ERROR';
    end;
end;

{---------------------------------------}
function ShortMonthIndex(inp: string): word;
var
	tmps: string;
begin
	tmps := trim(lowercase(inp));
    if length(tmps) > 3 then tmps := copy(tmps, 1, 3);
    Result := 0;
	if tmps = 'jan' then Result := 1;
    if tmps = 'feb' then Result := 2;
    if tmps = 'mar' then Result := 3;
    if tmps = 'apr' then Result := 4;
    if tmps = 'may' then Result := 5;
    if tmps = 'jun' then Result := 6;
    if tmps = 'jul' then Result := 7;
    if tmps = 'aug' then Result := 8;
    if tmps = 'sep' then Result := 9;
    if tmps = 'oct' then Result := 10;
    if tmps = 'nov' then Result := 11;
    if tmps = 'dec' then Result := 12;
end;

{---------------------------------------}
function DayIndex(inp: string): integer;
var
	tmps: string;
begin
	tmps := trim(lowercase(inp));
    if length(tmps) > 3 then tmps := Copy(tmps, 1, 3);
    Result := 0;
    if tmps = 'sun' then result := 1;
    if tmps = 'mon' then result := 2;
    if tmps = 'tue' then result := 3;
    if tmps = 'wed' then result := 4;
    if tmps = 'thu' then result := 5;
    if tmps = 'fri' then result := 6;
    if tmps = 'sat' then result := 7;
end;

{---------------------------------------}
function y2kadjust(inp: TDatetime): TDateTime;
begin
    {
    Check dates for y2k..
    If the date is < 1970 then adjust to 2000
    }

    if inp < Date1970 then
        Result := inp + DateY2K
    else
        Result := inp;
end;

{---------------------------------------}
procedure y2kFormat;
var
    fs, ds: string;
    idx: integer;
begin
    {
    adjust the ShortDateFormat var..
    to make sure we are using 4 digit years.
    }

    ds := lowercase(trim(ShortDateFormat));
    if pos('yyyy', ds) <= 0 then begin
        idx := pos('yy', ds);
        fs := Copy(ds, 0, idx-1);
        fs := fs + 'yy';
        fs := fs + Copy(ds, idx, length(ds)-idx+1);
        ds := fs;
        end;
    ShortDateFormat := ds;
end;

{---------------------------------------}
function GetFraction(inp: string): real;
var
	tmps: string;
    num, denom: string;
    p, v1, v2: integer;
begin
	{return a real values based on the input fraction}
    tmps := trim(inp);

    try
        p := Pos('/', tmps);
        num := LeftChar(tmps, p-1);
        denom := RightChar(tmps, length(tmps) - p);
        v1 := StrToInt(num);
        v2 := StrToInt(denom);

        if abs(v2) < 0.0001 then begin
            Result := -1;
            exit;
            end;

        Result := v1 / v2;
    except
    	Result := -1;
    end;
end;

{---------------------------------------}
function BigStrToDateTime(inp: string): TDateTime;
var
	i, tz: integer;
	mo, dy, yr: word;
	zone, tmps: string;
    plist: TStringList;
    time: TDateTime;
begin
	{
    convert internet style dates to a DateTime.
    Can be in the format:
    Wed, 29 Jul 1998 01:14:20 GMT
    29 Jul 1998 01:14:20 -0500
    etc..
    }

    // default...
    Result := Now;

    plist := TStringList.Create;
    pparse(' ', inp, plist);
    tmps := trim(plist[0]);
    if RightChar(tmps, 1) = ',' then tmps := LeftChar(tmps, length(tmps) -1);

    if DayIndex(tmps) > 0 then begin
        i := 1;
        end
    else
    	i := 0;

    while trim(plist[i])='' do inc(i);

    dy := StrToInt(plist[i]); inc(i);
    mo := ShortMonthIndex(plist[i]); inc(i);
    yr := StrToInt(plist[i]); inc(i);

    {
    IGNORE Time zones...
    Time Zones were causing sorting problems in most apps..
    Code removed.
    }
    try
    	tmps := plist[i];
        if length(tmps) > 8 then tmps := Copy(tmps, 1, 8);
        time := StrToTime(tmps); inc(i);
        if i < plist.count then
        	zone := plist[i]
        else
        	zone := 'GMT';

        {
        if zone = 'GMT' then tz := 0
        else if zone = 'EST' then tz := +5
        else if zone = 'CST' then tz := +6
        else if zone = 'MST' then tz := +7
        else if zone = 'PST' then tz := +8
        else if zone = 'EDT' then tz := +6
        else if zone = 'CDT' then tz := +7
        else if zone = 'MDT' then tz := +8
        else if zone = 'PDT' then tz := +9
        else begin
            if pos('+', zone) > 0 then tz := StrToInt(zone)
            else if pos('-', zone) > 0 then tz := StrToInt(zone)
            else tz := 0;
            tz := tz div 100;
            end;
        }

        tz := 0;
    	time := time + (tz / 24.0);
        if (yr < 1900) and (yr > 50) then yr := yr + 1900;
    	Result := EncodeDate(yr, mo, dy) + time;
    except
    end;
    plist.Free;
end;

{---------------------------------------}
function NumDaysInMonth(mo, yr: integer): integer;
begin
	case mo of
    1: Result := 31;
    2: Result := 28;
    3: Result := 31;
    4: Result := 30;
    5: Result := 31;
    6: Result := 30;
    7: Result := 31;
    8: Result := 31;
    9: Result := 30;
    10: Result := 31;
    11: Result := 30;
    12: Result := 31;
    else Result := -1;
    end;

    {check for leap years..}
    if mo = 2 then begin
    	if (yr mod 4) = 0 then inc(Result);
        end;
end;

{---------------------------------------}
procedure DelINIKey(fname, sect, key: string);
var
	pfn: array[0..255] of char;
    psect: array[0..255] of char;
    pkey: array[0..255] of char;
begin
	StrPCopy(pfn, fname);
    StrPCopy(psect, sect);
    StrPCopy(pkey, key);

    WritePrivateProfileString(psect, pkey, nil, pfn);
end;

{---------------------------------------}
procedure WriteFont(fnt: TFont; ini: TiniFile; sect, key: string);
var
    bold, uline, italic: boolean;
begin
	{write a TFont out to a ini File..}
    with fnt, ini do begin
    	WriteString(sect, key + '_name', Name);
        WriteInteger(sect, key + '_size', Size);
        if (fsBold in Style) then bold := true else bold := false;
        if (fsUnderline in Style) then uline := true else uline := false;
        if (fsItalic in Style) then italic := true else italic := false;
        WriteString(sect, key + '_bold', BoolToStr(bold));
        WriteString(sect, key + '_uline', BoolToStr(uline));
        WriteString(sect, key + '_italic', BoolToStr(italic));
        end;
end;

{---------------------------------------}
procedure ReadFont(fnt: TFont; ini: TiniFile; sect, key: string);
var
	bold, uline, italic: boolean;
begin
	{read a TFont from an ini file..}
    with fnt, ini do begin
    	Name := ReadString(sect, key + '_name', 'Arial');
        Size := ReadInteger(sect, key + '_size', 10);
        bold := StrToBool(ReadString(sect, key + '_bold', 'FALSE'));
        uline := StrToBool(ReadString(sect, key + '_uline', 'FALSE'));
        italic := StrToBool(ReadString(sect, key + '_italic', 'FALSE'));
        if bold then Style := Style + [fsBold];
        if uline then Style := Style + [fsUnderline];
        if italic then Style := Style + [fsItalic];
        end;
end;

{---------------------------------------}
function GetVersionInfo(KeyList, ValList: TStringList): string;
const
    InfoNum = 10;
    InfoStr : array [1..InfoNum] of String =
        ('CompanyName', 'FileDescription', 'FileVersion', 'InternalName',
        'LegalCopyright', 'LegalTradeMarks', 'OriginalFilename',
        'ProductName', 'ProductVersion', 'Comments');
{$ifdef Win32}
var
    S: String;
    n: DWORD;
    Len: UINT;
    i: Integer;
    Buf: PChar;
    Value: PChar;
{$endif}
begin
{$ifdef Win32}
    Result := '';
    KeyList.Clear;
    ValList.Clear;
    S := Application.ExeName;
    n := GetFileVersionInfoSize(PChar(S),n);
    if n > 0 then begin
        Buf := AllocMem(n);
        GetFileVersionInfo(PChar(S),0,n,Buf);
        if VerQueryValue(Buf,PChar('StringFileInfo\040904E4\'+ InfoStr[3]),Pointer(Value),Len) then
            Result := Value;
        for i:=1 to InfoNum do begin
            if VerQueryValue(Buf,PChar('StringFileInfo\040904E4\'+ InfoStr[i]),Pointer(Value),Len) then begin
                KeyList.Add(InfoStr[i]);
                ValList.Add(Value);
                end;
            end;
        FreeMem(Buf,n);
        end
    else
        Result := '';
{$endif}
end;

{---------------------------------------}
function NlogX(X: Real; N:Real): Real;
begin
  NlogX := Ln(X) / Ln(N);
end;

{---------------------------------------}
function BaseChange(Num, NewBase : Integer) : String;
Const
  BaseChars : Array [0..36] of Char = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
Var
  St : String;
begin
  St := '';
  Repeat
    St  := BaseChars[Num MOD NewBase] + St;
    Num := Num Div NewBase;
  Until Num = 0;
  BaseChange := St;
end;

{---------------------------------------}
function TrimPath(inp: string): string;
var
    tmps: string;
begin
    tmps := inp;
    while (RightChar(tmps, 1) = '\') do
        tmps := LeftChar(tmps, Length(tmps)-1);

    Result := tmps;
end;

{---------------------------------------}
procedure DoPrevInstance(winclass, appname, inifile: string; msg: integer);
var
    ihandle:    hwnd;
    sz:         word;
    ini:        TiniFile;
    ename:      string;
begin
    {
    This function attempts to create a sync. event using the
    appname. If it already exists, it writes a line
    to the ini specified and fires off
    the spec'd msg to the named class window
    }
    {$ifdef Win32}
    ename := appname + ' Event';
    CreateEvent(nil, false, false, PChar(appname));
    if GetLastError = ERROR_ALREADY_EXISTS then begin
        ihandle := FindWindow(PChar(winclass), nil);
        if IsWindow(iHandle) = false then begin
            ShowMessage('Error in IPC comms');
            Halt;
            end;
        sz := Length(ParamStr(1)) + 1;
        if sz = 1 then Halt;
        ini := TiniFile.Create(inifile);
        ini.WriteString('LOADFILES', 'load', ParamStr(1));
        ini.Free;
        if SendMessage(iHandle, msg, 0, 0) <> 0 then
            ShowMessage('Error posting msg');
        Halt;
        end;
    {$endif}
end;

{---------------------------------------}
procedure ClearList(lst: TList);
var
    i: integer;
begin
    {clear's each object in a TList}
    for i := lst.Count - 1 downto 0 do begin
        TObject(lst[i]).Free;
        lst.Delete(i);
        end;
end;

{---------------------------------------}
function WinNT: boolean;
{$ifdef Win32}
var
   WinVerInfo: OSVERSIONINFOA;
{$endif}
begin
    {check for WinNT}
    {$ifdef Win32}
    WinVerInfo.dwOSVersionInfoSize := SizeOf(WinVerInfo);
    GetVersionEx(WinVerInfo);
    with WinVerInfo do begin
        if dwPlatformID = VER_PLATFORM_WIN32_NT then
            Result := true
        else
            Result := false;
        end;
    {$else}
    Result := false;
    {$endif}
end;

{---------------------------------------}
function GetStringHeight(HDC: Integer; inp: string): integer;
var
    ts: TSize;
begin
    {returns the height of a string in pixels..}
    Result := -1;
    if hdc <= 0 then
        exit;

    ts.cx := 0;
    ts.cy := 0;
    {$ifdef Win32}
    if not GetTextExtentPoint32(HDC, PChar(inp), Length(inp), ts) then
        Result := -1
    else
        Result := ts.cy;
    {$else}
     Result := -1;
    {$endif}
end;

{---------------------------------------}
function IsValidIP(inp: string): boolean;
var
    tokens: TStringList;
    i: integer;
    cv: array[0..4] of longint;
begin
    {
    function evaluates the string to determine
    if it's a valid IP address in the form of
    123.123.123.124

    pgm 9/30/99 - new function.
    }
    Result := false;
    tokens := TStringList.Create;
    pparse('.', inp, tokens);
    if tokens.Count = 4 then begin
        try
            Result := true;
            for i := 0 to 3 do begin
                cv[i] := StrToInt(tokens[i]);
                if cv[i] <= 0 then Result := false;
                if cv[i] > 255 then Result := false;
                end;
        except
            Result := false;
        end;
        end;
    tokens.Free;
end;


{$ifdef Windows}
{---------------------------------------}
{
    These are some functions written
    which provide compatability with
    new functions in Win32. These should
    only be using when using Delphi 1,
    to compile Win16 apps
}
{---------------------------------------}
{---------------------------------------}
function CopyFile(src, dest: string): boolean;
var
	TempStream: TMemoryStream;
begin
	TempStream := TMemoryStream.Create;
    with TempStream do begin
    	LoadFromFile(src);
        SaveToFile(dest);
        Free;
        end;
    Result := true;
end;

{---------------------------------------}
function ExtractFileDrive(path: string): string;
begin
	{return the drive root.. like C:\}
    Result := LeftChar(path, 3);
end;

{---------------------------------------}
function DriveNum(path: string): integer;
var
	tmps: string;
begin
	{return 0=A, 1=B, etc...}
    tmps := Uppercase(path);
    Result := ord(tmps[1]) - 65;
end;

{---------------------------------------}
function Trim(inp: string): string;
var
	t1, t2: string;
    ex: boolean;
    i: integer;
begin
	{trim all left spaces...}
	t1 := '';
    ex := false;
    for i := 1 to length(inp) do begin
        if (inp[i] <> ' ') or (ex = true) then
        	begin
        	t1 := t1 + inp[i];
            ex := true;
            end;
    	end;
    t2 := t1;
    {trim all right spaces...}
    for i := length(t1) downto 2 do
    	begin
        if inp[i] = ' ' then
            t2 := Copy(t1, 1, i-1)
        else
        	break;
        end;
    Result := t2;
end;
{$endif}


{---------------------------------------}
{---------------------------------------}
{---------------------------------------}
{$ifdef TRAYAPP}
initialization
begin
	ShowWindow(Application.Handle, SW_HIDE);
end;
{$endif}


end.
